/*
This is shareware code which was taken from Larry Widing's bitmap.zip
file and given out to CS 146B at SJSU, October, 1996, by Rudy Rucker.
The bitmap.zip file was obtained from the "Shareware Code" page at
http://www.pfdpf.state.oh.us/msawczyn/files/
You can also get to it from the Borland C++ home page.
Rucker took the pieces needed for loading and saving bitmaps and
put them into this file, which is called dibsave.cpp.

Widing's Readme.doc includes this statement:

"If you have any questions on this code, suggestions for improvement, or
just to tell me you found it useful, you can contact me through my
CompuServe account number 70253,3461.

For further image processing information, I would suggest checking out
Martin Heller's Advanced Windows Programming book, which provided the
basis for my PCX and GIF file loading code.

As a final note, I have tried to make this code as problem free as
possible, but I make no guarantees.  Use at your own risk.

Larry Widing "
*/

#include "types.h" //for <windows.h>
#include	<malloc.h>
#include	<dos.h>
#include	<stdlib.h>
#include "error2.hpp"  //For the ErrorBox function
#include "types.h" //For the __FLAT__ switch
#include <stdio.h> //For unlink
/*
**	Types
*/
typedef	HANDLE	HDIB;		/* Handle to a DIB */
#ifndef __FLAT__ //16 bit case
	#ifndef	HUGE
	#define	HUGE	huge
	#endif //HUGE
#else //32 bit case defines "HUGE" to be an empty string 
	#define HUGE
#endif


/*
**	Global Variables
*/
extern HWND			main_hwnd;			/* Handle to the application's main window */

/*
**	Modification History
**	--------------------
**	$lgb$
** 10/15/91     Larry Widing   Initial version for Win Tech Journal Article.
** 11/12/91     Larry Widing   Added defines for print menu options and
**                             conversion menu.
** 02/10/92     Larry Widing   Added hooks to alter the colormap.  Added HDIB
**                             typdef.  Added hooks for reading compressed RIX
**                             images.
** 08/03/92     Larry Widing
**	$lge$
*/


/*----------------------------This is from bmfile.c----------------*/

/*
** int												number of color entries in header
** DIBitmapColors(BITMAPINFO FAR *bmi);	pointer to bitmap header
**
**    This function returns the number of colors in the color table of
**	the specified Device-Independant Bitmap.
**
** Modification History:
** 09/12/91  LCW  Created
*/
int
DIBitmapColors(BITMAPINFO FAR *bmi)
{
	if (bmi->bmiHeader.biSize == sizeof(BITMAPCOREHEADER))
	{
		/*
		**	OS/2 PM Bitmap, use bcBitCount field to determine colors
		*/
		switch (((BITMAPCOREINFO FAR *)bmi)->bmciHeader.bcBitCount)
		{
			case 1:
				return 2;	/* Monochrome bitmap -> 2 colors */

			case 4:
				return 16;	/* 4-bit image -> 16 colors */

			case 8:
				return 256;	/* 8-bit image -> 256 colors */

			case 24:
				return 0;	/* 24-bt image -> 0 colors in color table */
		}
	}
	else
	{
		/*
		**	Windows bitmap
		*/
		if (bmi->bmiHeader.biClrUsed == 0)
		{
			/* Maximum number of entries */
			switch (bmi->bmiHeader.biBitCount)
			{
				case 1:
					return 2;	/* Monochrome bitmap -> 2 colors */

				case 4:
					return 16;	/* 4-bit image -> 16 colors */

				case 8:
					return 256;	/* 8-bit image -> 256 colors */

				case 24:
					return 0;	/* 24-bt image -> 0 colors in color table */
			}
		}
		else
		{
			return (int)bmi->bmiHeader.biClrUsed;
		}
	}

	return 0;
}

LPSTR
DIBitmapBits(BITMAPINFO FAR *bmi)
{
	LPSTR	bits;
	int	colors = DIBitmapColors(bmi);

	bits = ((LPSTR)bmi) + (unsigned int)bmi->bmiHeader.biSize;

	if (bmi->bmiHeader.biSize == sizeof(BITMAPCOREHEADER))
	{
		bits += colors * sizeof(RGBTRIPLE);
	}
	else
	{
		bits += colors * sizeof(RGBQUAD);
	}

	return bits;
}


//-----------------READ-------------
/*
** LPSTR											pointer to bitmap bits
** DIBitmapBits(BITMAPINFO FAR *bmi);	pointer to bitmap header
**
**    This function returns a pointer to the bits in a packed Device-
**	Independant Bitmap.
**
** Modification History:
** 09/12/91  LCW  Created
*/

/*
** HDIB												handle of created DI bitmap
** ReadBitmapFile(const char *filename);	name of file to load
**
**    This function will read the passed file in, and create a device
**	dependant bitmap, returning the handle to the created bitmap to the
**	caller.
**
** Modification History:
** 09/06/91  LCW  Created
*/
HDIB
ReadDIBFile(char *filename)
{
	HFILE					file;
	int					rc;
	unsigned int		block;
	unsigned long		size;
	HDIB				hdata = (HANDLE)NULL;
	char HUGE			*pdata;
	char HUGE			*ptr;
	BITMAPFILEHEADER	bfHdr;
	OFSTRUCT				ofs;

	/*
	**	1. Open file
	*/
	file = OpenFile((LPSTR)filename, (LPOFSTRUCT)&ofs, OF_READ | OF_SHARE_DENY_WRITE);
	if (file != -1)
	{
		/*
		**	2. Read in BITMAPFILEHEADER and verify that this is a bitmap file
		*/
		rc = _lread(file, (LPSTR)&bfHdr, sizeof(BITMAPFILEHEADER));
		if (rc == sizeof(BITMAPFILEHEADER)
			|| bfHdr.bfType == ('B' + ('M' << 8)))
		{
			/*
			**	2.1. Verify that the bfSize field is correct
			*/
		{
			DWORD	_offset = _llseek(file, 0L, 1);
			DWORD	_size = _llseek(file, 0L, 2);
			_llseek(file, _offset, 0);
			if (bfHdr.bfSize != _size)
				bfHdr.bfSize = _size;
		}

			/*
			**	3. Allocate storage for packed DIB
			*/
			size = bfHdr.bfSize - sizeof(BITMAPFILEHEADER);
			hdata = GlobalAlloc(GMEM_MOVEABLE | GMEM_NODISCARD, size);
			if (hdata != (HANDLE)NULL)
			{
				rc = -1;
				pdata = (char HUGE *)GlobalLock(hdata);
				if (pdata != NULL)
				{
					/*
					**	4. Read in DIB header and bits into packed-DIB buffer
					*/
					block = 16 * 1024;	/* size of chunks to read in */
					ptr = pdata;

					while (size > 0)
					{
						if (size < (unsigned long)block)
						{
							block = (unsigned int)size;
						}

						if (_lread(file, (LPSTR)ptr, block) != block)
						{
							ErrorBox("LARRYDIB.C module", "ReadBitmapFile(): Error reading BMP file");
							break;
						}

						size -= (long)block;
#if	defined(__TSC__)
						if ((FP_OFF(ptr) + block) == 0)
						{
							ptr = MK_FP(FP_SEG(ptr) + 8, 0);
						}
						else
						{
							ptr += block;
						}
#else
						ptr += block;
#endif
					}

					if (size == 0)
					{
						rc = 0;
					}
					GlobalUnlock(hdata);
				}
				else
				{
					ErrorBox("LARRYDIB.C module", "ReadBitmapFile(): Error locking packed DIB memory");
				}

				if (rc < 0)
				{
					GlobalFree(hdata);
					hdata = (HANDLE)NULL;
				}
			}
			else
			{
				ErrorBox("LARRYDIB.C module", "ReadBitmapFile(): Unable to allocate memory for packed DIB");
			}
		}
		else
		{
			ErrorBox("LARRYDIB.C module", "ReadBitmapFile(): Error reading BITMAPFILEHEADER");
		}
		_lclose(file);
	}
	else
	{
		ErrorBox("LARRYDIB.C module", "ReadBitmapFile(): Unable to open bitmap file");
	}

	return hdata;
}
/*
** HBITMAP						handle of logical bitmap created, or NULL
** DibToBitmap(HDIB hbm);	handle of packed DIB to convert
**
**    This function converts a packed DIB into a logical bitmap, returning
**	a handle to the resulting bitmap.
**
** Modification History:
** 10/16/91  LCW  Created
*/

HBITMAP
DibToBitmap(HDIB hbm)
{
	HBITMAP			bm = (HBITMAP)NULL;
	HDC				hdc;
	BITMAPINFO FAR	*bmi;
	LPSTR				bits;

	bmi = (BITMAPINFO FAR *)GlobalLock(hbm);
	if (bmi != NULL)
	{
		hdc = GetDC(main_hwnd);
		if (hdc != (HDC)NULL)
		{
			bits = DIBitmapBits(bmi);

			bm = CreateDIBitmap(hdc, (LPBITMAPINFOHEADER)bmi, CBM_INIT,
				bits, bmi, DIB_RGB_COLORS);

			ReleaseDC(main_hwnd, hdc);
		}

		GlobalUnlock(hbm);
	}

	return bm;
}

/*
** HDIB					handle of new packed DIB (NULL if an error occurred)
** BitmapToDIB(
**   HBITMAP hbm,		handle of logical bitmap to convert
**   int mode);		mode to use when converting
**
**    This function will convert a logical bitmap into a packed Device
**	Independant Bitmap, using the mode parameter to determine if the resulting
**	DIB should be run length encoded (mode == 1), a OS/2 compatible DIB
**	(mode == 2) or a unencoded DIB (mode == 0).
**
** Modification History:
** 09/07/91  LCW  Created
*/
HDIB
BitmapToDIB(HBITMAP hbm, int mode)
{
	int				i;
	HDC				hdc;
	HDIB				result = (HDIB)NULL;
	BITMAPINFO		*bmi;
	BITMAPCOREINFO	*bmci;
	LPSTR				ptr, sptr, dptr;
	int				hdrSize;
	int				bitsPerPixel;
	BITMAP			bm;

	/*
	**	Get bitmap information
	*/
	GetObject(hbm, sizeof(bm), (LPSTR)&bm);
	if (bm.bmPlanes == 1)
		bitsPerPixel = bm.bmBitsPixel;
	else
		bitsPerPixel = bm.bmPlanes;


	if (mode == 2)
	{
		/*
		**	Building a OS/2 bitmap - allocate a LPBITMAPCOREINFO record
		*/
		hdrSize = sizeof(BITMAPCOREHEADER);
		switch (bitsPerPixel)
		{
			case 1:
				hdrSize += 2 * sizeof(RGBTRIPLE);
				break;

			case 3:
				++bitsPerPixel;
			case 4:
				hdrSize += 16 * sizeof(RGBTRIPLE);
				break;

			case 8:
				hdrSize += 256 * sizeof(RGBTRIPLE);
				break;
		}
		bmci = (BITMAPCOREINFO *)malloc(hdrSize);
		bmci->bmciHeader.bcSize = sizeof(BITMAPCOREHEADER);
		bmci->bmciHeader.bcWidth = (WORD)bm.bmWidth;
		bmci->bmciHeader.bcHeight = (WORD)bm.bmHeight;
		bmci->bmciHeader.bcPlanes = 1;
		bmci->bmciHeader.bcBitCount = bitsPerPixel;
		bmi = NULL;
	}
	else
	{
		/*
		**	Building a Windows compatible Bitmap
		*/
		hdrSize = sizeof(BITMAPINFOHEADER);

		switch (bitsPerPixel)
		{
			case 1:
				hdrSize += 2 * sizeof(RGBQUAD);
				break;

			case 3:
				++bitsPerPixel;
			case 4:
				hdrSize += 16 * sizeof(RGBQUAD);
				break;

			case 8:
				hdrSize += 256 * sizeof(RGBQUAD);
				break;
		}
		bmi = (BITMAPINFO *)malloc(hdrSize);

		bmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
		bmi->bmiHeader.biWidth = bm.bmWidth;
		bmi->bmiHeader.biHeight = bm.bmHeight;
		bmi->bmiHeader.biPlanes = 1;
		bmi->bmiHeader.biBitCount = bitsPerPixel;
		if (mode == 0 || (bitsPerPixel != 8 && bitsPerPixel != 4))
			bmi->bmiHeader.biCompression = BI_RGB;
		else if (bitsPerPixel == 8)
			bmi->bmiHeader.biCompression = BI_RLE8;
		else
			bmi->bmiHeader.biCompression = BI_RLE4;
		bmi->bmiHeader.biSizeImage = 0;
		bmi->bmiHeader.biXPelsPerMeter = 0;
		bmi->bmiHeader.biYPelsPerMeter = 0;
		bmi->bmiHeader.biClrUsed = 0;
		bmi->bmiHeader.biClrImportant = 0;
		bmci = NULL;
	}

	/*
	**	Get a DC to use
	*/
	hdc = GetDC(main_hwnd);
	if (hdc != (HDC)NULL)
	{
		/*
		**	Allocate storage needed
		*/
		if (bmi == NULL)
		{
			DWORD	bitSize = (DWORD)bm.bmWidth * (DWORD)bitsPerPixel / 8;
			if ((bitSize & 3) != 0)
				bitSize += 4 - (bitSize & 3);

			result = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
				(long)hdrSize + (DWORD)bm.bmHeight * bitSize);

			if (result != (HANDLE)NULL)
			{
				ptr = (LPSTR)GlobalLock(result);
				if (ptr == NULL)
				{
					GlobalFree(result);
					result = (HANDLE)NULL;
					ErrorBox("LARRYDIB.C module", "BitmapToDIB(): Unable to lock DIB memory");
				}
				else
				{
					sptr = (LPSTR)bmci;
				}
			}
		}
		else
		{
			if (GetDIBits(hdc, hbm, 0, bm.bmHeight, NULL, (LPBITMAPINFO)bmi, DIB_RGB_COLORS))
			{
				result = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
					(long)hdrSize + bmi->bmiHeader.biSizeImage);

				if (result != (HANDLE)NULL)
				{
					ptr = (LPSTR)GlobalLock(result);
					if (ptr == NULL)
					{
						GlobalFree(result);
						result = (HANDLE)NULL;
						ErrorBox("LARRYDIB.C module", "BitmapToDIB(): Unable to lock DIB memory");
					}
					else
					{
						sptr = (LPSTR)bmi;
					}
				}
			}
		}

		if (result)
		{
			/*
			**	Copy header
			*/
			dptr = ptr;
			for (i = 0 ; i < hdrSize ; ++i)
			{
				*dptr++ = *sptr++;
			}

			/*
			**	Get the bits
			*/
			if (!GetDIBits(hdc, hbm, 0, bm.bmHeight, dptr, (LPBITMAPINFO)ptr, DIB_RGB_COLORS))
			{
				GlobalUnlock(result);
				GlobalFree(result);
				result = (HANDLE)NULL;
			}
			else
			{
				GlobalUnlock(result);
			}
		}
		ReleaseDC(main_hwnd, hdc);
	}
	else
	{
		ErrorBox("LARRYDIB.C module", "BitmapToDIB(): Unable to get DC from main window");
	}

	if (bmi != NULL)
		free(bmi);

	if (bmci != NULL)
		free(bmci);

	return result;
}

HBITMAP ReadBitmapFile(char *filename)
{
	return DibToBitmap(ReadDIBFile(filename));
}

//----------------WRITE-----------------



/*
** int
** WriteBitmapFile(
**		const char *filename,	name of file to load
**		const HDIB hbm);		handle to packed DIB
**
**    This function will write the passed packed Device Independant Bitmap
**	to the specified file.
**
** Modification History:
** 09/06/91  LCW  Created
*/


int
WriteDIBitmapFile(char *filename, HDIB hbm)
{
	int					file;
	int					rc = -1;
	unsigned int		block;
	unsigned long		size;
	char HUGE			*ptr;
	BITMAPFILEHEADER	bfHdr;
	LPBITMAPINFO		bmi;
	OFSTRUCT				ofs;

	/*
	**	1. Open output file
	*/
	file = OpenFile((LPSTR)filename, (LPOFSTRUCT)&ofs, OF_CREATE | OF_WRITE);
	if (file != -1)
	{
		/*
		**	2. Lock memory resource
		*/
		bmi = (LPBITMAPINFO)GlobalLock(hbm);
		if (bmi != NULL)
		{
			/*
			**	3. Create BITMAPFILEHEADER and write to file
			*/
			bfHdr.bfType = ('B' + ('M' << 8));
			bfHdr.bfSize = sizeof(BITMAPFILEHEADER) + GlobalSize(hbm);
			bfHdr.bfReserved1 = 0;
			bfHdr.bfReserved2 = 0;
			bfHdr.bfOffBits = sizeof(BITMAPFILEHEADER)
				+ (DWORD)(DIBitmapBits(bmi) - (LPSTR)bmi);
			if (_lwrite(file, (LPSTR)&bfHdr, sizeof(bfHdr)) == sizeof(bfHdr))
			{
				/*
				**	4. Write out DIB header and packed bits to file
				*/
				size = GlobalSize(hbm);
				ptr = (char HUGE *)bmi;
				block = 16 * 1024;	/* size of chunks to write out */

				while (size > 0)
				{
					if (size < (unsigned long)block)
					{
						block = (unsigned int)size;
					}

					if (_lwrite(file, (LPSTR)ptr, block) != block)
					{
						ErrorBox("LARRYDIB.C module", "WriteBitmapFiile(): Error writing DIB");
						break;
					}

					size -= (long)block;
#if	defined(__TSC__)
					if ((FP_OFF(ptr) + block) == 0)
					{
						ptr = MK_FP(FP_SEG(ptr) + 8, 0);
					}
					else
					{
						ptr += block;
					}
#else
					ptr += block;
#endif
				}

				if (size == 0)
				{
					rc = 0;
				}
			}
			else
			{
				ErrorBox("LARRYDIB.C module", "WriteBitmapFile(): Error writing BITMAPFILEHEADER");
			}

			GlobalUnlock(hbm);
		}
		else
		{
			ErrorBox("LARRYDIB.C module", "WriteBitmapFile(): Error locking bitmap into memory");
		}
		_lclose(file);
	}
	else
	{
		ErrorBox("LARRYDIB.C module", "WriteBitmapFile(): Error opening output file");
	}

	if (rc != 0)
	{
		unlink(filename);
	}

	return rc;
}


int
WriteBitmapFile(char *filename, HBITMAP hbm)
{
	return WriteDIBitmapFile(filename, BitmapToDIB(hbm, 1));
}


